//
//
// diskusage
//
//
// Author: Christophe Pallier (pallier@lscp.ehess.fr)
//
// 
//     Copyright (C) 1997  Christophe Pallier 
//     
//     This program is free software; you can redistribute it and/or
//     modify it under the terms of the GNU General Public License
//     as published by the Free Software Foundation; either version 2
//     of the License, or (at your option) any later version.
//     
//     This program is distributed in the hope that it will be useful,
//     but WITHOUT ANY WARRANTY; without even the implied warranty of
//     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//     GNU General Public License for more details.
//     
//     You should have received a copy of the GNU General Public License
//     along with this program; if not, write to the Free Software
//     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

import java.lang.Runnable;
import java.io.PrintStream;
import java.io.File;

/** 
 * A class that recursively scans a directory and its sub_dirs,
 * printing the cumulated size of all files.<br>
 * The class implements Runnable and the output PrintStream is passed to the constructor
 * to allow to use this within a Thread in a larger program.  
 * @author Christophe Pallier (pallier@lscp.ehess.fr)
 * @version 2.0 date 31/10/97
 */
public class diskusage implements Runnable
{
	private String topdirname=null;
	private int blksize=1; // block (=cluster) size in bytes
	private PrintStream out=null; // output stream
	private int div=1; // divider for displaying file sizes
	private int pos=8; // width of the 'size' field

	public diskusage(String dirname,int blocksize,PrintStream sout) {
		topdirname=dirname;
		blksize=blocksize;
		out=sout;
	}

	/** The main loop. */
	public void run() {
		recurse_dir(topdirname);
		out.flush();
		if (out!=System.out) { out.close(); }
	}

	/** sets a divider for displaying files' sizes. */
	public void setdiv(int d) { div=d; }
	
	/** sets width of the size field in the output. */
	public void setpos(int w) { pos=w; }

	private String JustifyRight(long num) {
		Long l=new Long(num);
		String s=l.toString();
		int nw=pos-s.length();
		StringBuffer m=new StringBuffer(pos+1);
		for (int i=0;i<nw;i++) m.append(" ");
		m.append(s);
		return new String(m);
	}

	/**
	 * Returns the space occupied by a file in a file system that allocates integer numbers of blocks.<br>
	 * return (blocksize?((filesize/blocksize+1)*blocksize):filesize).
	 */
	 protected long bsize(long filesize) {
		if (blksize>0) { return (filesize / blksize + 1)*blksize; }
		else { return filesize; }
	}

	/** 
	 * Returns the total size (in bytes) of files in dirname.<br>
	 * As a side-effect, if the Hashtable 'dirtable' is not null, 
	 * pairs of subdirectory names and sizes are put in it.<br>
	 *
	 * @param dirname	the directory to browse
	 * @param blocksize  size of blocks in the file system (see bsize())
	 * @see bsize
	 */
	
	
	public long recurse_dir(String dirname) {
		File dir=new File(dirname);
		try { // Security Exception
			if (dir.isFile()) return bsize(dir.length());
			if (dir.isDirectory()) {
				String[] subfiles=dir.list();
				// why this can arise '(e.g. on '/'), i don't know (!?)
				if (subfiles==null) { 
                    System.err.println("Problem accessing [" + dirname + "] (To access, try [\\\\.] in lieu of [\\])"); 
					return 0; 
				}

				long tsize=0L;
				long fsize=0L;
				for (int i=0;i<subfiles.length;i++) {
					fsize = recurse_dir(dirname+File.separatorChar+subfiles[i]);
					if (fsize>0) tsize +=fsize;
				}
				out.println(JustifyRight((tsize+bsize(dir.length()))/div)+"\t"+dirname);
				return tsize+bsize(dir.length());
			}
			// it is neither a file nor a directory !
			System.err.println("Cannot access [" + dirname + "]"); 
			return 0; 

		} catch (SecurityException e) { 
			System.err.println(dirname + ": access restricted"); 
			return 0; 
		}
	}

	public static void printUsage() {
			System.out.println("Usage:    java diskusage   path   [block_size]\n");
			System.out.println("Lists the sizes in Kb occupied by the subdirs of 'path'.");
			System.out.println("'block_size' is the size in bytes of clusters (or blocks) used");
			System.out.println("by the filesystem to allocate space on disk.");
			System.out.println("\nExample:     java diskusage .. 16384 | sort -r");
			System.out.println("\n                      Author: Christophe Pallier (pallier@lscp.ehess.fr)\n");
			System.out.println("A bug in some java for Windows requires typing [\\\\.] rather than [\\] for root!");
	}

	public static void main(String[] args) {
		if (args.length==0) { printUsage(); return; }
		if ((args[0]=="-?")||(args[0]=="-h")) { printUsage(); return; }
		
		String dir=args[0];
		int blksz=1;
		if (args.length>1) { blksz=Integer.parseInt(args[1]); }
		diskusage du=new diskusage(dir,blksz,System.out);
		du.setdiv(1024);
		Thread ta=new Thread(du);
		ta.start();
	}
}

